<?php
/**
 * Plugin Name: Dewata AI Chatbot
 * Plugin URI: https://dewataai.com
 * Description: Tambahkan AI Chatbot Dewata ke website WordPress Anda dengan mudah. Cukup masukkan API key dan chatbot langsung aktif!
 * Version: 1.0.1
 * Author: Dewata AI
 * Author URI: https://dewataai.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: dewata-chatbot
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Prevent direct file access
if (!defined('WPINC')) {
    die;
}

// Define plugin constants
define('DEWATA_CHATBOT_VERSION', '1.0.1');
define('DEWATA_CHATBOT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DEWATA_CHATBOT_PLUGIN_DIR', plugin_dir_path(__FILE__));

/**
 * Add settings menu
 */
function dewata_chatbot_add_menu() {
    add_options_page(
        'Dewata AI Chatbot',
        'Dewata Chatbot',
        'manage_options',
        'dewata-chatbot',
        'dewata_chatbot_settings_page'
    );
}
add_action('admin_menu', 'dewata_chatbot_add_menu');

/**
 * Register settings
 */
function dewata_chatbot_register_settings() {
    register_setting('dewata_chatbot_settings', 'dewata_chatbot_api_key', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => ''
    ));
    
    register_setting('dewata_chatbot_settings', 'dewata_chatbot_exclude_pages', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_textarea_field',
        'default' => ''
    ));
    
    register_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_home', array(
        'type' => 'boolean',
        'default' => true
    ));
    
    register_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_posts', array(
        'type' => 'boolean',
        'default' => true
    ));
    
    register_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_pages', array(
        'type' => 'boolean',
        'default' => true
    ));
    
    register_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_archives', array(
        'type' => 'boolean',
        'default' => true
    ));
}
add_action('admin_init', 'dewata_chatbot_register_settings');

/**
 * Settings page HTML
 */
function dewata_chatbot_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        
        <div style="background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%); color: white; padding: 20px; border-radius: 8px; margin: 20px 0;">
            <h2 style="color: white; margin-top: 0;">🤖 Dewata AI Chatbot</h2>
            <p style="font-size: 16px; margin-bottom: 0;">Chatbot AI yang siap melayani pelanggan Anda 24/7. Dapatkan API Key di <a href="https://dewataai.com/dashboard" target="_blank" style="color: #fbbf24;">dewataai.com/dashboard</a></p>
        </div>
        
        <form method="post" action="options.php">
            <?php settings_fields('dewata_chatbot_settings'); ?>
            
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="dewata_chatbot_api_key">API Key</label>
                    </th>
                    <td>
                        <input 
                            type="text" 
                            id="dewata_chatbot_api_key" 
                            name="dewata_chatbot_api_key" 
                            value="<?php echo esc_attr(get_option('dewata_chatbot_api_key')); ?>" 
                            class="regular-text"
                            placeholder="Masukkan API Key dari dashboard Dewata AI"
                        />
                        <p class="description">Dapatkan API Key dari <a href="https://dewataai.com/dashboard/bots" target="_blank">Dashboard → Bot Saya → API Keys</a></p>
                    </td>
                </tr>
            </table>
            
            <h2 class="title">Pengaturan Tampilan</h2>
            <p>Pilih di mana chatbot akan ditampilkan:</p>
            
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">Tampilkan di Homepage</th>
                    <td>
                        <label>
                            <input type="checkbox" name="dewata_chatbot_show_on_home" value="1" <?php checked(get_option('dewata_chatbot_show_on_home', true)); ?> />
                            Tampilkan chatbot di halaman utama
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Tampilkan di Single Post</th>
                    <td>
                        <label>
                            <input type="checkbox" name="dewata_chatbot_show_on_posts" value="1" <?php checked(get_option('dewata_chatbot_show_on_posts', true)); ?> />
                            Tampilkan chatbot di halaman artikel/post
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Tampilkan di Pages</th>
                    <td>
                        <label>
                            <input type="checkbox" name="dewata_chatbot_show_on_pages" value="1" <?php checked(get_option('dewata_chatbot_show_on_pages', true)); ?> />
                            Tampilkan chatbot di halaman statis (pages)
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Tampilkan di Archives</th>
                    <td>
                        <label>
                            <input type="checkbox" name="dewata_chatbot_show_on_archives" value="1" <?php checked(get_option('dewata_chatbot_show_on_archives', true)); ?> />
                            Tampilkan chatbot di halaman kategori, tag, dan arsip
                        </label>
                    </td>
                </tr>
            </table>
            
            <h2 class="title">Exclude Halaman Tertentu</h2>
            
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="dewata_chatbot_exclude_pages">Halaman yang Dikecualikan</label>
                    </th>
                    <td>
                        <textarea 
                            id="dewata_chatbot_exclude_pages" 
                            name="dewata_chatbot_exclude_pages" 
                            rows="5" 
                            class="large-text code"
                            placeholder="Masukkan ID atau slug halaman, satu per baris. Contoh:&#10;123&#10;about-us&#10;contact"
                        ><?php echo esc_textarea(get_option('dewata_chatbot_exclude_pages')); ?></textarea>
                        <p class="description">
                            Masukkan ID halaman (angka) atau slug halaman, satu per baris.<br>
                            Chatbot tidak akan ditampilkan di halaman-halaman ini.<br>
                            <strong>Contoh:</strong> <code>123</code> atau <code>contact-us</code>
                        </p>
                    </td>
                </tr>
            </table>
            
            <?php submit_button('Simpan Pengaturan'); ?>
        </form>
        
        <div style="background: #f0f0f1; padding: 15px; border-radius: 4px; margin-top: 20px;">
            <h3 style="margin-top: 0;">❓ Butuh Bantuan?</h3>
            <p>Kunjungi <a href="https://dewataai.com/dashboard/tutorial" target="_blank">Tutorial</a> atau hubungi kami di <a href="mailto:contact@dewatatech.com">contact@dewatatech.com</a></p>
        </div>
    </div>
    <?php
}

/**
 * Check if current page should show chatbot
 */
function dewata_chatbot_should_show() {
    // Check page type settings
    if (is_front_page() || is_home()) {
        if (!get_option('dewata_chatbot_show_on_home', true)) {
            return false;
        }
    } elseif (is_single()) {
        if (!get_option('dewata_chatbot_show_on_posts', true)) {
            return false;
        }
    } elseif (is_page()) {
        if (!get_option('dewata_chatbot_show_on_pages', true)) {
            return false;
        }
    } elseif (is_archive() || is_category() || is_tag()) {
        if (!get_option('dewata_chatbot_show_on_archives', true)) {
            return false;
        }
    }
    
    // Check exclude list
    $exclude_pages = get_option('dewata_chatbot_exclude_pages', '');
    if (!empty($exclude_pages)) {
        $excludes = array_map('trim', explode("\n", $exclude_pages));
        $excludes = array_filter($excludes); // Remove empty lines
        
        $current_id = get_queried_object_id();
        $current_slug = '';
        
        // Get current page/post slug
        $queried_object = get_queried_object();
        if ($queried_object && isset($queried_object->post_name)) {
            $current_slug = $queried_object->post_name;
        }
        
        foreach ($excludes as $exclude) {
            // Check by ID (numeric)
            if (is_numeric($exclude) && intval($exclude) === $current_id) {
                return false;
            }
            // Check by slug
            if (!empty($current_slug) && $exclude === $current_slug) {
                return false;
            }
        }
    }
    
    return true;
}

/**
 * Inject chatbot widget script
 */
function dewata_chatbot_inject_widget() {
    $api_key = get_option('dewata_chatbot_api_key');
    
    // Don't inject if no API key
    if (empty($api_key)) {
        return;
    }
    
    // Don't inject in admin
    if (is_admin()) {
        return;
    }
    
    // Check if should show on current page
    if (!dewata_chatbot_should_show()) {
        return;
    }
    
    // Output the widget script
    ?>
    <!-- Dewata AI Chatbot Widget -->
    <script src="https://dewataai.com/widget.js" data-api-key="<?php echo esc_attr($api_key); ?>" async></script>
    <?php
}
add_action('wp_footer', 'dewata_chatbot_inject_widget', 100);

/**
 * Add settings link on plugin page
 */
function dewata_chatbot_settings_link($links) {
    $settings_link = '<a href="options-general.php?page=dewata-chatbot">Pengaturan</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'dewata_chatbot_settings_link');

/**
 * Plugin activation hook
 */
function dewata_chatbot_activate() {
    // Set default options on activation
    add_option('dewata_chatbot_api_key', '');
    add_option('dewata_chatbot_exclude_pages', '');
    add_option('dewata_chatbot_show_on_home', true);
    add_option('dewata_chatbot_show_on_posts', true);
    add_option('dewata_chatbot_show_on_pages', true);
    add_option('dewata_chatbot_show_on_archives', true);
}
register_activation_hook(__FILE__, 'dewata_chatbot_activate');

/**
 * Plugin deactivation hook
 */
function dewata_chatbot_deactivate() {
    // Clean up any transients or temp data if needed
    // Options are kept for re-activation
}
register_deactivation_hook(__FILE__, 'dewata_chatbot_deactivate');

/**
 * Plugin uninstall - clean up all options from database
 * This runs when the plugin is deleted from WordPress
 */
function dewata_chatbot_uninstall() {
    // Remove all plugin options from database
    delete_option('dewata_chatbot_api_key');
    delete_option('dewata_chatbot_exclude_pages');
    delete_option('dewata_chatbot_show_on_home');
    delete_option('dewata_chatbot_show_on_posts');
    delete_option('dewata_chatbot_show_on_pages');
    delete_option('dewata_chatbot_show_on_archives');
    
    // Clean up any registered settings
    unregister_setting('dewata_chatbot_settings', 'dewata_chatbot_api_key');
    unregister_setting('dewata_chatbot_settings', 'dewata_chatbot_exclude_pages');
    unregister_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_home');
    unregister_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_posts');
    unregister_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_pages');
    unregister_setting('dewata_chatbot_settings', 'dewata_chatbot_show_on_archives');
}
register_uninstall_hook(__FILE__, 'dewata_chatbot_uninstall');
